/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package structuring.matcher;


import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Map;

import structuring.Translator;
import util.Combinations;
import util.StmtUtil;
import ast.expression.Expression;
import ast.expression.UnaryExpression;
import ast.expression.UnaryOp;
import ast.statement.DoWhile;
import ast.statement.Statement;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.statement.JumpStmt;

/*
 *  |/\
 *  0  |
 *  |\/
 *  |
 *  1
 */

public class BusyWaitMatcher implements Pattern {
  private ArrayList<ArrayList<BasicBlock>> vertices = new ArrayList<ArrayList<BasicBlock>>();
  private ArrayList<ArrayList<BbEdge>>     edges    = new ArrayList<ArrayList<BbEdge>>();


  public ArrayList<ArrayList<BbEdge>> edges() {
    return edges;
  }


  public int internalEdges() {
    return 1;
  }


  public ArrayList<ArrayList<BasicBlock>> vertices() {
    return vertices;
  }


  public int internalVertices() {
    return 1;
  }

  public boolean match(BasicBlock pred) {
    if (pred.getOutlist().size() != 2) {
      return false;
    }
    ArrayList<BbEdge> outlist = new ArrayList<BbEdge>(pred.getOutlist());
    Combinations combinations = new Combinations(2, outlist.size());

    while (combinations.hasNext()) {
      ArrayList<Integer> comb = combinations.next();

      BbEdge backedge = outlist.get(comb.get(0));

      BasicBlock pred2 = backedge.getDst();
      BasicBlock exit = outlist.get(comb.get(1)).getDst();

      if (pred != pred2) {
        continue;
      }

      addToVerices(pred, exit);
      addToEdges(backedge);
    }
    return !vertices.isEmpty();
  }

  private void addToVerices(BasicBlock pred, BasicBlock exit) {
    ArrayList<BasicBlock> list = new ArrayList<BasicBlock>(2);
    list.add(pred);
    list.add(exit);
    vertices.add(list);
  }

  private void addToEdges(BbEdge backedge) {
    ArrayList<BbEdge> list = new ArrayList<BbEdge>(1);
    list.add(backedge);
    edges.add(list);
  }


  public JumpStmt getNewJump(ArrayList<BasicBlock> config) {
    return new JumpStmt(config.get(1).getId());
  }


  public Statement getStmtCode(ArrayList<BasicBlock> config, Map<BasicBlock, Statement> mapping) {
    assert (vertices.contains(config));
    LinkedList<cfg.statement.Statement> stmt = config.get(0).getCode();
    assert (StmtUtil.getNrOfStmts(stmt) == 1);
    Expression expr = Translator.translate(StmtUtil.toJump(stmt.getLast()).getExpression());
    if (isInverted(config)) {
      expr = new UnaryExpression(expr, UnaryOp.Not);
    }

    Statement body = mapping.get(config.get(0));
    return new DoWhile(body, expr);
  }

  private boolean isInverted(ArrayList<BasicBlock> config) {
    BasicBlock pred  = config.get(0);
    
    ArrayList<BbEdge> ine = new ArrayList<BbEdge>(pred.getInlist());
    assert (ine.size() == 2);
    BbEdge edge;
    if (ine.get(0).getSrc() == pred) {
      edge = ine.get(0);
    } else {
      assert (ine.get(1).getSrc() == pred);
      edge = ine.get(1);
    }
    assert( edge.getSrc() == pred );
    assert( edge.getDst() == pred );
    return edge.getNr() == 0;
  }

}
