/*
 * Part of upcompiler. Copyright (c) 2012, Urs Fässler, Licensed under the GNU Genera Public License, v3
 * @author: urs@bitzgi.ch
 */

package structuring.matcher;


import java.util.ArrayList;
import java.util.Map;

import structuring.Translator;
import util.Combinations;
import util.StmtUtil;
import ast.expression.Expression;
import ast.statement.BlockStmt;
import ast.statement.IfStmt;
import ast.statement.Statement;
import cfg.basicblock.BasicBlock;
import cfg.basicblock.BbEdge;
import cfg.statement.JumpStmt;

/*
 *   0
 *  / \
 * 1   2
 *  \ /
 *   3
 */

public class IfThenElseMatcher implements Pattern {
  private ArrayList<ArrayList<BasicBlock>> vertices = new ArrayList<ArrayList<BasicBlock>>();
  private ArrayList<ArrayList<BbEdge>>     edges    = new ArrayList<ArrayList<BbEdge>>();


  public ArrayList<ArrayList<BbEdge>> edges() {
    return edges;
  }


  public int internalEdges() {
    return 3;
  }


  public ArrayList<ArrayList<BasicBlock>> vertices() {
    return vertices;
  }


  public int internalVertices() {
    return 3;
  }

  public boolean match(BasicBlock pred) {
    ArrayList<BbEdge> outlist = new ArrayList<BbEdge>(pred.getOutlist());
    if (outlist.size() != 2) {
      return false;
    }
    Combinations combinations = new Combinations(2, outlist.size());

    while (combinations.hasNext()) {
      ArrayList<Integer> comb = combinations.next();

      BbEdge predThen = outlist.get(comb.get(0));
      BbEdge predElse = outlist.get(comb.get(1));
      BasicBlock then = predThen.getDst();
      BasicBlock els_ = predElse.getDst();

      if (then == pred) {
        continue;
      }

      if (els_ == pred) {
        continue;
      }
      
      if( predThen.getNr() != 1 ){
        continue;
      }

      if (then.getInlist().size() != 1) { // TODO allow this if insertion of goto is needed
        continue;
      }

      if (els_.getInlist().size() != 1) { // TODO allow this if insertion of goto is needed
        continue;
      }

      if (then.getOutlist().size() != 1) {
        continue;
      }

      if (els_.getOutlist().size() != 1) {
        continue;
      }

      BbEdge thenExit = then.getOutlist().iterator().next();
      BasicBlock joinT = thenExit.getDst();
      BasicBlock joinE = els_.getOutlist().iterator().next().getDst();

      if (joinT == joinE) {
        addToVertices(pred, then, els_, joinT);
        addToEdges(predThen, predElse, thenExit);
      }
    }
    return !vertices.isEmpty();
  }

  private void addToVertices(BasicBlock predicate, BasicBlock then, BasicBlock els_, BasicBlock join) {
    ArrayList<BasicBlock> list = new ArrayList<BasicBlock>(4);
    list.add(predicate);
    list.add(then);
    list.add(els_);
    list.add(join);
    vertices.add(list);
  }

  private void addToEdges(BbEdge predThen, BbEdge predElse, BbEdge thenExit) {
    ArrayList<BbEdge> list = new ArrayList<BbEdge>(4);
    list.add(predThen);
    list.add(predElse);
    list.add(thenExit);
    edges.add(list);
  }


  public JumpStmt getNewJump(ArrayList<BasicBlock> config) {
    return new JumpStmt(config.get(3).getId());
  }


  public Statement getStmtCode(ArrayList<BasicBlock> config, Map<BasicBlock, Statement> mapping) {
    assert (vertices.contains(config));

    Expression expr =  Translator.translate(StmtUtil.toJump(config.get(0).getCode().getLast())
        .getExpression());
    assert (StmtUtil.getNrOfStmts(config.get(1).getCode()) == 1);
    assert (StmtUtil.getNrOfStmts(config.get(2).getCode()) == 1);
    assert (config.get(1).getCode().getLast() instanceof JumpStmt);
    assert (config.get(2).getCode().getLast() instanceof JumpStmt);

    BlockStmt block = new BlockStmt();
    IfStmt ifs = new IfStmt(expr, mapping.get(config.get(1)), mapping.get(config.get(2)));
    block.addCode(mapping.get(config.get(0)));
    block.addCode(ifs);
    return block;
  }

}
