/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file DataLogger.cpp
 *
 * @brief Implements the DataLogger.
 * @author Nicolas Rüegg, Urs Fässler
 */

#include "DataLogger.h"

using namespace std;

/**
 * The constructor.
 * @param fileName Path to the logfile.
 * @param loglevel Loglevel. The loglevel is determined by OR-ing the message types.
 * @see MessageType
 */
DataLogger::DataLogger(string fileName, int loglevel )
{
	// set the loglevel (OR combination of MessageType values)
	m_loglevel = loglevel;

	m_logFile.open( fileName.c_str() );

	if( !( m_logFile.is_open() ) )
	{
		throw new runtime_error( "could not open logfile" +fileName );
	}
}


/**
 * The destructor.
 * Cleans up. Closes open files, streams and so on.
 */
DataLogger::~DataLogger() {
  m_logFile.flush();
	m_logFile.close();
}


/**
 * Writes the type of a log message into the logfile.
 * Usually as the first field of a new line.
 *
 * @param type Type of the message.
 * @see MessageType
 */
void DataLogger::writeType(MessageType type) {
	switch (type)
	{
	case INFO:
		m_logFile << "INFO" << SEPERATOR;
		break;
	case RESULT:
		m_logFile << "RESULT" << SEPERATOR;
		break;
	case WARN:
		m_logFile << "WARN" << SEPERATOR;
		break;
	case ERROR:
		m_logFile << "ERROR" << SEPERATOR;
		break;
	default:
		break;
	}
}


/**
 * Write a message of the specified type to the logfile.
 * INFO messages are written to stdout and ERROR to stderr, additionally.
 * @param type The type of the message.
 * @param text The message to write to the logfile.
 */
void DataLogger::write( MessageType type, string text )
{
	// write WARNINGS and ERRORS to stderr anyways
	// and INFO to stdout
	if ( type == WARN || type == ERROR ) {
		cerr << text << endl;
	} else if (type == INFO) {
		cout << text << endl;
	}

	if ( m_loglevel & type  ) {
		// write the type of the message as the first field
		writeType(type);

		m_logFile << text << endl;
	}
}



/**
 * Write a string and a corresponding value separated by a ";".
 *
 * @param name	Identifier of the following value.
 * @param value	A value to the identifier.
 */
void DataLogger::writeValue( string name, double value )
{
	m_logFile << name << SEPERATOR << value << SEPERATOR;
}



/**
 * Write an entire map to the logfile or console as a message of the specified type.
 * @param type The type of the message.
 * @param values The map to write to the logfile.
 */
void DataLogger::write( MessageType type, const map<string,double>* values )
{
	// just log if the loglevel is set accordingly
	if ( m_loglevel & type  ) {
		// write the type of the message as the first field of this line
		writeType(type);

		map<string,double>::const_iterator   itr;

		for( itr = values->begin(); itr != values->end(); itr++)
		{
			writeValue( itr->first, itr->second );
		}
		m_logFile << endl;
	}

}


/**
 * Write two maps one after another to the logfile.
 * Intended for maps which are directly related to one another and therefore
 * should be written to the same line (like the parameters and the corresponding results).
 * @param parameters First map. Map of the parameters.
 * @param results Second map. Map of the results.
 */
void DataLogger::write( const map<string,double>* parameters, const map<string,double>* results )
{
	// just log if the loglevel is set accordingly
	if ( m_loglevel & RESULT ) {
		// write the type of the message as the first field of this line
		writeType(RESULT);

		map<string,double>::const_iterator   itr;

		// write parameters first...
		for( itr = parameters->begin(); itr != parameters->end(); itr++)
		{
			writeValue( itr->first, itr->second );
		}

		// ... then write results.
		for( itr = results->begin(); itr != results->end(); itr++)
		{
			writeValue( itr->first, itr->second );
		}
		m_logFile << endl;
	}

}

