/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file DataLogger.h
 *
 * @brief Header of the DataLogger.
 */

#ifndef DATALOGGER_H_
#define DATALOGGER_H_

#include <map>
#include <string>
#include <fstream>
#include <iostream>
#include <stdexcept>

using namespace std;

/**
 * The MessageType enum.
 * The MessageType defines which message types are available.
 * The values are set the way, that they can be OR-ed together
 * to get a loglevel (to specify which message types should be logged).
 */
enum MessageType {NONE = 0, INFO = 1, RESULT = 2, WARN = 4, ERROR = 8};

/**
 * @class DataLogger
 * @brief DataLogger writes data into a logfile and to the console.
 *
 * The DataLogger provides methods to write log messages of a specified type into a
 * logfile. There's one line for every message and the fields of a message are
 * separated by a ";". The first field of a message is the message type.
 *
 * INFO messages are written to stdout, additionally. ERROR messages to stderr.
 *
 * @author Nicolas Rüegg, Urs Fässler
 *
 * @see MessageType
 */
class DataLogger {
private:
  /// Logfile.
  ofstream   m_logFile;
  /// Current loglevel.
  int	m_loglevel;

  void writeType(MessageType type);

  void writeValue( string name, double value );

public:
	DataLogger(std::string fileName, int loglevel);
	virtual ~DataLogger();

	void write( MessageType type, std::string text );
	void write( MessageType type, const std::map<std::string,double>* values );
	void write( const std::map<std::string,double>* parameters, const std::map<std::string,double>* results );

	static const char SEPERATOR  = ';';
};

#endif /* DATALOGGER_H_ */
