/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file Experimenter.cpp
 *
 * @brief Implements the Experimenter.
 * @author Nicolas Rüegg, Urs Fässler
 */

#include "Experimenter.h"

/**
 * Constructor.
 * Sets the configuration file, the logfile and the loglevel for this series
 * of experiments.
 *
 * @param configFile Path to the config file.
 * @param logFile Path to the logfile.
 * @param loglevel The loglevel for this series of experiments.
 */
Experimenter::Experimenter( string& configFile, string& logFile, int loglevel )
{
	// create DataLogger
	m_log = new DataLogger::DataLogger( logFile, loglevel );

	initLoader();
	loadConfig( configFile );
}


/**
 * Initialize the specific experiment loaders;
 */
void Experimenter::initLoader()
{
  AlgorithmExperimentLoader*   alexlo   = new AlgorithmExperimentLoader();
  SimulationExperimentLoader*  siexlo   = new SimulationExperimentLoader();
  ArrayExperimentLoader*       arexlo   = new ArrayExperimentLoader();

  alexlo->add( "Evolution", new EvolutionaryAlgorithmLoader( m_log ) );
  alexlo->add( "Simulated Annealing", new SimulatedAnnealingAlgorithmLoader( m_log ) );
  alexlo->add( "Textbook Simulated Annealing", new TextbookSaAlgorithmLoader( m_log ) );
  alexlo->add( "Systematic", new SystematicAlgorithmLoader( m_log ) );

  m_experimentLoader.addExperimentLoader( "algorithm", alexlo );
  m_experimentLoader.addExperimentLoader( "simulation", siexlo );
  m_experimentLoader.addExperimentLoader( "array", arexlo );

  m_experimentLoader.setLogger( m_log );
}


/**
 * Load the configuration file and initialize the root experiment.
 */
void Experimenter::loadConfig( string& configFile )
{
  xmlpp::DomParser parser;

  try
  {
    parser.parse_file( configFile );
  }
  catch( xmlpp::exception* e )
  {
    throw new LoaderException( "read configuration file: " + string( e->what() ) );
  }

  ExperimentSettings  settings;

  settings.seed   = 0;


  m_experimentLoader.readSettings( parser.get_document()->get_root_node(), &settings );

  if( settings.seed == 0 )
  {
    settings.seed    = time( NULL );
    m_log->write( INFO, "Using generated seed: " + intToStr( settings.seed ) );
  }
  else
  {
    m_log->write( INFO, "Using seed from configuration: " + intToStr( settings.seed ) );
  }

  srand( settings.seed );

  m_experiment  = m_experimentLoader.createExperiment( m_experimentLoader.findExperiment( parser.get_document()->get_root_node() ) );
}


/**
 * Destructor.
 * Cleans up. Closes open files and so on.
 */
Experimenter::~Experimenter() {
	delete m_log;
}


/**
 * Conducts the series of experiments.
 * Starts the series of experiments by running the root experiment.
 *
 * @param path Defines the root of the path to which the intermediate results are written.
 * @return The fitness value.
 */
double Experimenter::run( Directory path )
{
  MapParameterToValueT  leer;

  return  m_experiment->run( &leer, path );
}
