/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef ALGORITHM_H_
#define ALGORITHM_H_

#include  "../../DataLogger/DataLogger.h"
#include  "../../common/include/types.h"
#include  "../../common/Directory.h"

/**
 * @class Algorithm
 * @brief Base class for all algorithm implementations.
 *
 * The usage of the methodes is as followed:
 * @code
 *   initialize
 *   while hasNext
 *   {
 *     next
 *     getName
 *     extendDirectory
 *     <run>
 *     fitnessFeedback
 *   }
 * @endcode
 */
class Algorithm
{
  private:
    DataLogger::DataLogger*   m_logger;

  protected:
    DataLogger::DataLogger* getLogger();

  public:
    Algorithm( DataLogger::DataLogger* log );

    /**
     * For setup of the algorithm.
     * @param changeable The parameter which can be changed
     */
    virtual void initialize( const MapParameterToRangeT* changeable ) = 0;

    /**
     * Returns true if the algorithm hasn't finished yet
     * @return
     */
    virtual bool hasNext() const = 0;

    /**
     * Returns a new map of parameter for the next step.
     * @return
     */
    virtual MapParameterToValueT next() = 0;

    /**
     * Extends the directory with a unique name.
     * @param dir
     */
    virtual void extendDirectory( Directory& dir ) const = 0;

    /**
     * Sets the fitness value for the current step.
     * @param fitness
     */
    virtual void fitnessFeedback( double fitness ) = 0;

    /**
     * Returns the name of the algorithm.
     * @return
     */
    virtual string getName() const = 0;

};


#endif /* ALGORITHM_H_ */
