/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

#include  "AlgorithmExperiment.h"

/**
 * Sets the whole bunch of member variables from the argument list.
 * @param algorithm
 * @param experiment
 * @param changeable
 * @param fixed
 * @param log
 * @return
 */
AlgorithmExperiment::AlgorithmExperiment
    (
      Algorithm*                        algorithm,
      Experiment*                       experiment,
      const map<string,ParamConfig>*    changeable,
      const map<string,ValueConfig>*    fixed,
      FitnessCalculation                fitness,
      DataLogger::DataLogger *log
    ): Experiment::Experiment( fitness, log )
{
   m_algorithm      = algorithm;
   m_experiment     = experiment;
   m_changeable     = changeable;
   m_fixed          = fixed;
}

/**
 * Runs an algorithm until it is finished. Directory for the sub experiment is
 * extended. The values from the parent experiment are merged with the fixed
 * values and the new values from algorithm.
 * @see Algorithm
 * @param value
 * @param path
 * @return
 */
double AlgorithmExperiment::run( const MapParameterToValueT* value, Directory path )
{
  map<string,double>      fixed;
  double                  fitness   = 0;
  MapParameterToRangeT    range;

  m_directory = path;

  fixed    = merge( makeMap( *m_fixed ), *value );

  range   = makeRange( *m_changeable );
  m_algorithm->initialize( &range );

  while( m_algorithm->hasNext() )
  {
    MapParameterToValueT  map;
    Directory             subdir( path );

    map       = m_algorithm->next();
    subdir.addSubdirectory( m_algorithm->getName() );
    m_algorithm->extendDirectory( subdir );

    map    = merge( fixed, map, *m_changeable );

    fitness   = m_experiment->run( &map, subdir );
    m_algorithm->fitnessFeedback( fitness );

    addFitness( fitness, &map );
  }

  return  exitExperiment();
}

/**
 * Creates from the configuration values a value map.
 * @param values
 * @return
 */
MapParameterToValueT AlgorithmExperiment::makeMap( const map<string,ValueConfig>& values ) const
{
  MapParameterToValueT                    result;
  map<string,ValueConfig>::const_iterator itr;

  for( itr = values.begin(); itr != values.end(); itr++ )
  {
    result[itr->first]  = itr->second.value;
  }

  return  result;
}

/**
 * Returns a merged map out of base and additional. Values from base are
 * overwritten by the value from additional.
 * @param base
 * @param additional
 * @return
 */
map<string,double> AlgorithmExperiment::merge( const map<string,double>& base, const map<string,double>& additional ) const
{
  MapParameterToValueT                    result;
  MapParameterToValueT::const_iterator    itr;

  result  = base;

//  result.insert( additional.begin(), additional.end() );
  for( itr = additional.begin(); itr != additional.end(); itr++ )
  {
    result[itr->first]  = itr->second;
  }

  return  result;
}

/**
 * Returns a merged map out of base and additional. Merge rules are used from
 * mode.
 * @param base
 * @param additional
 * @param mode is used to know how base and additional should be merged.
 * @return
 */
map<string,double> AlgorithmExperiment::merge( const map<string,double>& base, const map<string,double>& additional, const map<string,ParamConfig>& mode ) const
{
  MapParameterToValueT                    result;
  MapParameterToValueT::const_iterator    itr;

  result  = base;

  for( itr = additional.begin(); itr != additional.end(); itr++ )
  {
    map<string,double>::const_iterator        baseValue;
    map<string,ParamConfig>::const_iterator   modeRule;
    string                                    key;

    key         = itr->first;
    baseValue   = base.find( key );
    modeRule    = mode.find( key );

    if( modeRule == mode.end() )
    {
      throw new runtime_error( "no mode for parameter " + key );
    }

    switch( modeRule->second.mode )
    {
      case PM_ABSOLUTE:
      {
        result[key]   = itr->second;
        break;
      }

      case PM_RELATIVE:
      {
        if( baseValue == base.end() )
        {
          throw new runtime_error( "for relative parameter, the parameter should exists (" + key + ")" );
        }
        result[key]   = itr->second + baseValue->second;
        break;
      }
    }
  }

  return  result;
}

/**
 * Returns a range map out of a configuration.
 * @param in
 * @return
 */
MapParameterToRangeT AlgorithmExperiment::makeRange( const map<string,ParamConfig>& in ) const
{
  MapParameterToRangeT                      result;
  map<string,ParamConfig>::const_iterator   itr;

  for( itr = in.begin(); itr != in.end(); itr++ )
  {
    RangeS  range;

    range.maximum     = itr->second.max;
    range.minimum     = itr->second.min;

    result[itr->first]    = range;
  }

  return  result;
}

string AlgorithmExperiment::getName() const
{
  return  m_algorithm->getName();
}
