/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

#include  "AlgorithmExperimentLoader.h"

/**
 * Frees all AlgorithmLoader.
 * @return
 */
AlgorithmExperimentLoader::~AlgorithmExperimentLoader()
{
  map<string,const AlgorithmLoader*>::const_iterator    idx;

  for( idx = m_algorithm.begin(); idx != m_algorithm.end(); idx++ )
  {
    delete  idx->second;
  }
}

/**
 * Adds a new algorithm loader by a specific name. The
 * AlgorithmExperimentLoader is than the owner of the loader.
 * @param name
 * @param loader
 */
void AlgorithmExperimentLoader::add( string name, const AlgorithmLoader* loader )
{
  map<string,const AlgorithmLoader*>::const_iterator    idx;

  idx   = m_algorithm.find( name );

  if( idx != m_algorithm.end() )
  {
    throw new LoaderException( "Algorithm \"" + name + "\" already exists." );
  }

  m_algorithm[name]   = loader;
}

/**
 * Returns the algorithm loader specified by name. If the loader is not found,
 * a LoaderException is thrown.
 * @param name
 * @return
 */
const AlgorithmLoader* AlgorithmExperimentLoader::get( string name ) const
{
  map<string,const AlgorithmLoader*>::const_iterator    idx;

  idx   = m_algorithm.find( name );

  if( idx == m_algorithm.end() )
  {
    throw new LoaderException( "Algorithm \"" + name + "\" not found." );
  }

  return  idx->second;
}

/**
 * Parse a string to the mode value.
 * @param in
 * @return
 */
ParamMode AlgorithmExperimentLoader::strToMode( string in ) const
{
  if( (in == "") || (in == "absoulte") )
  {
    return  PM_ABSOLUTE;
  }
  else if( in == "relative" )
  {
    return  PM_RELATIVE;
  }
  else
  {
    throw new LoaderException( "unknown parameter mode: " + in );
  }
}

/**
 * Read one parameter from the xml node.
 * @param node
 * @param map
 */
void AlgorithmExperimentLoader::readParam( const xmlpp::Node* node, map<string,ParamConfig>* map ) const
{
  ParamConfig             range;
  string                  name;
  const xmlpp::Element*   elem    = dynamic_cast<const xmlpp::Element*>( node );

  range.min     = strToFloat( elem->get_attribute_value( "min", "" ) );
  range.max     = strToFloat( elem->get_attribute_value( "max", "" ) );
  range.mode    = strToMode( elem->get_attribute_value( "mode", "" ) );
  name          = elem->get_attribute_value( "name", "" );

  (*map)[name]  = range;
}

/**
 * Read one fixed parameter from the xml node.
 * @param node
 * @param map
 */
void AlgorithmExperimentLoader::readFixed( const xmlpp::Node* node, map<string,ValueConfig>* map ) const
{
  ValueConfig             value;
  string                  name;
  const xmlpp::Element*   elem    = dynamic_cast<const xmlpp::Element*>( node );

  value.value     = strToFloat( elem->get_attribute_value( "value", "" ) );
  value.mode      = strToMode( elem->get_attribute_value( "mode", "" ) );
  name            = elem->get_attribute_value( "name", "" );

  (*map)[name]    = value;
}

/**
 * Read all parameter from parameter section.
 * @param node
 * @param changeable
 */
void AlgorithmExperimentLoader::readParamsChangeable( const xmlpp::Node* node, map<string,ParamConfig>* changeable ) const
{
  xmlpp::Node::NodeList             nodes   = node->get_children( "param" );
  xmlpp::Node::NodeList::iterator   itr;

  for( itr = nodes.begin(); itr != nodes.end(); itr++ )
  {
    readParam( *itr, changeable );
  }
}

/**
 * Read all fixed parameter from parameter section.
 * @param node
 * @param fixed
 */
void AlgorithmExperimentLoader::readParamsFixed( const xmlpp::Node* node, map<string,ValueConfig>* fixed ) const
{
  xmlpp::Node::NodeList             nodes   = node->get_children( "fixed" );
  xmlpp::Node::NodeList::iterator   itr;

  for( itr = nodes.begin(); itr != nodes.end(); itr++ )
  {
    readFixed( *itr, fixed );
  }
}

/**
 * Returns a map with all parameters from the experiment.
 * @param experiment
 * @return
 */
map<string,ParamConfig>* AlgorithmExperimentLoader::loadChangeable( const xmlpp::Element* experiment ) const
{
  const xmlpp::Node::NodeList           nodes       = experiment->get_children( "parameters" );

  map<string,ParamConfig>*              changeable  = new map<string,ParamConfig>();

  if( nodes.size() == 1 )
  {
    readParamsChangeable( *nodes.begin(), changeable );
  }

  return  changeable;
}

/**
 * Returns a map with all fixed parameters from the experiment.
 * @param experiment
 * @return
 */
map<string,ValueConfig>* AlgorithmExperimentLoader::loadFixed( const xmlpp::Element* experiment ) const
{
  const xmlpp::Node::NodeList           nodes       = experiment->get_children( "parameters" );

  map<string,ValueConfig>*              fixed       = new map<string,ValueConfig>();

  if( nodes.size() == 1 )
  {
    readParamsFixed( *nodes.begin(), fixed );
  }

  return  fixed;
}

/**
 * Creates a algorithm as specified by the xml node.
 * @param experiment
 * @return
 */
Algorithm* AlgorithmExperimentLoader::loadAlgorithm( const xmlpp::Element* experiment ) const
{
  const AlgorithmLoader*          loader;
  string                          algorithm;
  algorithm         = experiment->get_attribute_value( "type", "" );
  loader            = get( algorithm );

  return loader->load( experiment );
}

/**
 * Create a algorithm experiment with a specific algorithm as specified in the
 * xml node.
 * @param experiment
 * @return
 */
Experiment* AlgorithmExperimentLoader::createExperiment( const xmlpp::Element* experiment ) const
{
  AlgorithmExperiment*    result;

  result    = new AlgorithmExperiment
  (
      loadAlgorithm( experiment ),
      getLoader()->createExperiment( getLoader()->findExperiment( experiment ) ),
      loadChangeable( experiment ),
      loadFixed( experiment ),
      readFitness( experiment, FC_MAXIMUM ),
      getLoader()->getLogger()
  );

  return  result;
}

