/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file EvolutionaryAlgorithm.cpp
 *
 * @brief Implements the EvolutionaryAlgorithm.
 * @author Nicolas Rüegg, Urs Fässler
 */


#include "EvolutionaryAlgorithm.h"
#include <iostream>

/**
 * Sorting criteria for the vector with the individuals.
 * @param left
 * @param right
 * @return true if left is logically less than right
 */
bool compareIndividual( const Individual& left, const Individual& right )
{
  return  left.fitness > right.fitness;
}

/**
 * The constructor.
 * @param numberOfGenerations	Number of generations.
 * @param populationSize		Size of the population.
 * @param numberOfParents		Number of parents (used for parent selection by LEADERS algorithm only)
 * @param changeProbability		Probability that a parameter value is altered from its parent's parameter value.
 * @param parameterScaling		Scaling parameter for the gaussian distributed changes to the parents value.
 * @param elitismOn				true secures the survival of the best individual of the current generation.
 * @param parentSelection		Chooses the parent selection algorithm.
 * @param distribution			Currently not used.
 * @param log					The data logger to use for data output.
 *
 * @see ParentSelectionAlgorithmT
 * @see DataLogger
 */
EvolutionaryAlgorithm::EvolutionaryAlgorithm
(
  int numberOfGenerations,
  uint populationSize,
  int numberOfParents,
  double changeProbability,
  bool elitismOn,
  ParentSelectionAlgorithmT parentSelection,
  Distribution distribution,
  double parameterScaling,
  DataLogger::DataLogger *log
) : Algorithm::Algorithm( log )
{
  initializeRandomNumberGenerators();

  m_numberOfGenerations     = numberOfGenerations;
  m_populationSize          = populationSize;
  m_numberOfParents         = numberOfParents;
  m_changeProbability       = changeProbability;
  m_elitismOn               = elitismOn;
  m_parentSelection         = parentSelection;
  m_distribution            = distribution;
  m_parameterScaling        = parameterScaling;
}


/**
 * Destructor.
 * Cleans up. Closes open files, streams and so on.
 */
EvolutionaryAlgorithm::~EvolutionaryAlgorithm()
{
  gsl_rng_free( m_randomNumberGenerator );
}


/**
 * Initializes the algorithm.
 * @param changeable	Map of changeable parameters.
 */
void EvolutionaryAlgorithm::initialize( const MapParameterToRangeT* const changeable )
{
  m_changeable  = *changeable;

  m_generationNumber = 1;
  createFirstGeneration();

  m_individualNumber  = 0;
  m_bestFitness       = 0;
}

/**
 * Tests if there's another individual.
 *
 * @return true when another individual is existing.
 */
bool EvolutionaryAlgorithm::hasNext() const
{
  return  (m_generationNumber < m_numberOfGenerations) || (m_individualNumber < m_generation.size());
}


/**
 * Sets the fitness value of the current individual.
 * Has to be set by the experiment owning this algorithm based on the return
 * value returned by the experiment's run() method.
 *
 * @param fitness	Fitness value achieved by the current individual.
 */
void EvolutionaryAlgorithm::fitnessFeedback( double fitness )
{
  m_bestFitness   = max( m_bestFitness, fitness );
  m_generation[m_individualNumber].fitness  = fitness;
  m_individualNumber++;
}


/**
 * Returns the next individual.
 * If all individuals of a generation are tested, a new generation is created automatically.
 * @return	The next individual.
 */
MapParameterToValueT EvolutionaryAlgorithm::next()
{
  // TODO test evolutionary experiment

  if( m_individualNumber >= m_generation.size() )
  {
    createNewGeneration();
    m_generationNumber++;

    m_individualNumber  = 0;
  }

  return m_generation[m_individualNumber].values;
}


/**
 * Expand the directory by "/evolution/generation<n>/individual<k>"
 * @param dir	Current directory path.
 */
void EvolutionaryAlgorithm::extendDirectory( Directory& dir ) const
{
  dir.addSubdirectory( "generation" + intToStr( m_generationNumber ) );
  dir.addSubdirectory( "individual" + intToStr( m_individualNumber ) );
}

/**
 * @return
 */
string EvolutionaryAlgorithm::getName() const
{
  return   "evolution";
}

/**
 * Initialize the random number generators.
 */
void EvolutionaryAlgorithm::initializeRandomNumberGenerators()
{
  // initialize random number generators
  // rand() has been initialized in the experimenter.
  m_randomNumberGenerator   = gsl_rng_alloc( gsl_rng_default );
  gsl_rng_set( m_randomNumberGenerator, rand() );
}


/**
 * Creates the initial generation with uniformly distributed parameter values.
 */
void EvolutionaryAlgorithm::createFirstGeneration()
{
  // kill current generation
  m_generation.clear();

  for( uint i = 0; i < m_populationSize; i++ )
  {
    // create individual (a set of parameters with their values)
    MapParameterToValueT individual;

    // get iterator over the parameters to optimize
    MapParameterToRangeT::const_iterator it_parametersToOptimize;

    // add generated values of parametersToOptimize to the map of parameterValues
    for( it_parametersToOptimize = m_changeable.begin(); it_parametersToOptimize != m_changeable.end(); it_parametersToOptimize++ )
    {
      // generate random value for this parameter
      double rangeMinimum = it_parametersToOptimize->second.minimum;
      double rangeMaximum = it_parametersToOptimize->second.maximum;
      double value = ((rangeMaximum - rangeMinimum) * gsl_rng_uniform( m_randomNumberGenerator )) + rangeMinimum;

      // write this to individual
      individual[it_parametersToOptimize->first]  = value;
    }

    Individual  neu;

    neu.fitness = -1;
    neu.values  = individual;

    m_generation.push_back( neu );
    // add individual to the first generation (fitness value is 0).

  }
}


/**
 * Creates an child based on the given parent.
 *
 * @param parent	Set of parameters of the parent.
 * @return Individual based on parent
 */
MapParameterToValueT EvolutionaryAlgorithm::createIndividual( const MapParameterToValueT* parent ) const
{
  MapParameterToValueT                  result;
  MapParameterToRangeT::const_iterator  itr;

  for( itr = m_changeable.begin(); itr != m_changeable.end(); itr++ )
  {
    double    random;
    double    value;
    string    param;

    param   = itr->first;
    value   = parent->find( param )->second;

    random  = gsl_rng_uniform( m_randomNumberGenerator );

    if( random <= m_changeProbability )
    {
      // Parameter is mutated. Compute how much.
      value = computeGaussianDistributedValue( value, itr->second.minimum, itr->second.maximum );
    }

    result[param]   = value;
  }

  return  result;
}


/**
 * Creates a new generation (parameter values gaussian distributed from the parents values).
 */
void EvolutionaryAlgorithm::createNewGeneration()
{
  // TODO: test createNewGeneration

  vector<Individual>   new_generation;

  sort( m_generation.begin(), m_generation.end(), compareIndividual );

  // TODO test elitism
  if( m_elitismOn )
  {
    new_generation.push_back( m_generation.front() );
  }

  // create the new generation
  while( new_generation.size() < m_populationSize )
  {
    Individual  neu; // = { m_fixed, -1 };
    neu.fitness = -1;
    neu.values  = createIndividual( selectParent() );
    new_generation.push_back( neu );
  }

  m_generation = new_generation;
}


/**
 * Find a random value in the range [minRange,maxRange] gaussian distributed around the given parentValue
 *
 * @param parentValue	The parent's value.
 * @param minRange		Minimal value.
 * @param maxRange		Maximal value.
 * @return Gauss distributed value.
 */
double EvolutionaryAlgorithm::computeGaussianDistributedValue( double parentValue, double minRange, double maxRange ) const
{
  // TODO test computeGaussianDistributedValue
  // what's rangeFactor for?
  double range = maxRange - minRange;
  double rangeFactor = range * m_parameterScaling;
  double childValue = 0.0;

  do
  {
    double randomNumber = gsl_ran_ugaussian( m_randomNumberGenerator );
    childValue = parentValue + (randomNumber * rangeFactor);
  }
  while( (childValue <= minRange) || (childValue >= maxRange) );

  return childValue;
}

/**
 * Selects a and returns a parent.
 *
 * @return parent
 */
const MapParameterToValueT* EvolutionaryAlgorithm::selectParent() const
{
  // the parent
  const MapParameterToValueT *parent = NULL;

  switch( m_parentSelection )
  {
    case LEADERS:
      // TODO test leaders algorithm
      parent = selectParentLeaders();
      break;
    case ROULETTE:
      // TODO test roulette algorithm
      parent = selectParentRoulette();
      break;
  }

  return parent;
}


/**
 * Selects a parent based on the roulette wheel algorithm.
 * A entirely random parent is selected every time this method is called.
 *
 * @return parent
 */
const MapParameterToValueT* EvolutionaryAlgorithm::selectParentRoulette() const
{
  // randomly return one of the m_populationSize individuals as the parent
  // idea of the algorithm:
  // 1. generate a random number between 0 and the added fitness of all
  //    individuals in a generation (total fitness).
  // 2. the parent is the one with whom the sum of the total fitness reaches
  //    the random number.

  // sum up all fitness values
  double total_fitness = 0.0;
  vector<Individual>::const_iterator it_generation;
  for( it_generation = m_generation.begin(); it_generation != m_generation.end(); it_generation++ )
  {
    total_fitness += it_generation->fitness;
  }

  // generate a random number in the range (0,total_fitness]
  double randomNumber = (total_fitness) * gsl_rng_uniform_pos( m_randomNumberGenerator );
  double tmp_fitness = 0.0;

  for( it_generation = m_generation.begin(); it_generation != m_generation.end(); it_generation++ )
  {
    tmp_fitness   = tmp_fitness + it_generation->fitness;

    if( tmp_fitness >= randomNumber )
    {
      return  &it_generation->values;
    }
  }

  throw new runtime_error( "EvolutionaryAlgorithm::selectParentRoulette" );
}


/**
 * Selects a parent based on the leaders algorithm.
 *
 * @return parent
 */
const MapParameterToValueT* EvolutionaryAlgorithm::selectParentLeaders() const
{
  // idea of the algorithm:
  //  1. create a vector with all the individuals of the generation
  //  2. sort the vector by the fitness value of the individuals
  //  3. randomly choose one of the first m_numberOfParents individuals as the parent


  // first m_numberOfParents individuals are possible parents
  double total_fitness = 0.0; // total fitness of the first m_numberOfParents individuals

  vector<Individual>::const_iterator it_generation = m_generation.begin();
  for( int i = 0; i < m_numberOfParents; i++ )
  {
    if( it_generation == m_generation.end() )
    {
      break;
    }
    total_fitness += it_generation->fitness;
    it_generation++;
  }

  // generate a random number in the range (0,total_fitness]
  double randomNumber = (total_fitness) * gsl_rng_uniform_pos( m_randomNumberGenerator );

  // and select the parent
  double tmp_fitness = 0.0;

  for( it_generation = m_generation.begin(); it_generation != m_generation.end(); it_generation++ )
  {
    tmp_fitness   = tmp_fitness + it_generation->fitness;

    if( tmp_fitness >= randomNumber )
    {
      return  &it_generation->values;
    }
  }

  throw new runtime_error( "EvolutionaryAlgorithm::selectParentLeaders" );
}
