/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file EvolutionaryAlgorithm.h
 *
 * @brief Header of the EvolutionaryAlgorithm.
 */

#ifndef EVOLUTIONARYALGORITHM_H_
#define EVOLUTIONARYALGORITHM_H_

#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <vector>
#include <algorithm>

#include  "../Algorithm.h"
#include  "../../../common/include/types.h"
#include  "../../../DataLogger/DataLogger.h"
#include  "../../../common/convert.h"

/**
 * selection algorithms
 */
typedef enum
{
  ROULETTE,
  LEADERS
} ParentSelectionAlgorithmT;

/**
 * Distribution of the individuals.
 */
typedef enum
{
  UNIFORM,
  GAUSSIAN
} Distribution;

/**
 * A single individual.
 */
typedef struct
{
  MapParameterToValueT    values;
  double                  fitness;
} Individual;

/**
 * @class EvolutionaryAlgorithm
 * @brief Evolutionary Algorithm implements an evolutionary algorithm.
 *
 * The EvolutionaryAlgorithm implements an evolutionary algorithm.
 * It creates a generation of indivduals and selects the parents from
 * this generation to create the next generation.
 * Selection of the parents can be done in two ways:
 * ROULETTE selects the parents with the roulette wheel algorithm,
 * whereas LEADERS just chooses the best individuals to be parents of the
 * next generation.
 *
 * @author Nicolas Rüegg, Urs Fässler
 *
 * @see Experiment
 * @see Algorithm
 * @see DataLogger
 */
class EvolutionaryAlgorithm: public Algorithm
{
  private:
	/// A generation of individuals.
    vector<Individual>            m_generation;
    /// Number of generations.
    unsigned int                  m_numberOfGenerations;
    /// Size of the population.
    uint                          m_populationSize;
    /// Number of parents (just used in LEADERS selection algorithm)
    int                           m_numberOfParents;
    /// Probability that a parameter value is altered from its parent's parameter value.
    double                        m_changeProbability;
    /// Scaling parameter for the gaussian distributed changes to the parents value.
    double						  m_parameterScaling;
    /// Elitism On secures the survival of the best individual of the current generation.
    bool                          m_elitismOn;
    /// Selects the parent selection algorithm.
    ParentSelectionAlgorithmT     m_parentSelection;
    /// Currently not used.
    Distribution                  m_distribution;
    /// Map of changeable parameters.
    MapParameterToRangeT          m_changeable;
    /// Number of the current generation.
    unsigned int                  m_generationNumber;
    /// Number of the current individual within the generation.
    unsigned int                  m_individualNumber;
    /// The random number generator.
    gsl_rng*                      m_randomNumberGenerator;

    double                        m_bestFitness;


    void initializeRandomNumberGenerators();
    double computeGaussianDistributedValue( double parentValue, double minRange, double maxRange ) const;

    MapParameterToValueT createIndividual( const MapParameterToValueT* parent ) const;
    void createFirstGeneration();
    void createNewGeneration();

    const MapParameterToValueT* selectParent() const;
    const MapParameterToValueT* selectParentRoulette() const;
    const MapParameterToValueT* selectParentLeaders() const;

  public:

    EvolutionaryAlgorithm( int numberOfGenerations, uint populationSize, int numberOfParents, double changeProbability, bool elitismOn, ParentSelectionAlgorithmT parentSelection, Distribution distribution, double parameterScaling, DataLogger::DataLogger* log  );
    virtual ~EvolutionaryAlgorithm();

    void initialize( const MapParameterToRangeT* const changeable );
    void extendDirectory( Directory& dir ) const;
    bool hasNext() const;
    MapParameterToValueT next();
    void fitnessFeedback( double fitness );
    string getName() const;

    static const  int           DEFAULT_NUMBER_OF_PARENTS         = 8;
    static const  int           DEFAULT_NUMBER_OF_GENERATIONS     = 100;
    static const  int           DEFAULT_POPULATION_SIZE           = 80;
    static const  bool          DEFAULT_ELITISM_ON                = false;
    static const  int           DEFAULT_CHANGE_PROBABILITY_1_OVER = 10;     // compiler don't allow double in static const
    static const  ParentSelectionAlgorithmT DEFAULT_PARENT_SELECTION    = ROULETTE;
    static const  Distribution  DEFAULT_DISTRIBUTION              = GAUSSIAN;
    static const  int           DEFAULT_PARAMETER_SCALING_1_OVER  = 1;
};

#endif /* EVOLUTIONARYALGORITHM_H_ */
