/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/


#include  "EvolutionaryAlgorithmLoader.h"

/**
 * Parse a string to "on" "off"
 * @param val
 * @return
 */
bool strToOnOff( string val )
{
  if( val == "on" )
  {
    return  true;
  }
  else if( val == "off" )
  {
    return  false;
  }
  else
  {
    throw new LoaderException( val + " is not a valid on/off value" );
  }
}

/**
 * Parse a string for ParentSelectionAlgorithmT.
 * @param val
 * @return
 */
ParentSelectionAlgorithmT strToSelection( string val )
{
  map<string,ParentSelectionAlgorithmT>   list;

  list["roulette"]    = ROULETTE;
  list["leaders"]     = LEADERS;

  return  list[val];
}

/**
 * Parse a string for Distribution.
 * @param val
 * @return
 */
Distribution strToDistribution( string val )
{
  map<string,Distribution>   list;

  list["uniform"]    = UNIFORM;
  list["gaussian"]   = GAUSSIAN;

  return  list[val];
}

/**
 * @see AlgorithmLoader
 * @param log
 * @return
 */
EvolutionaryAlgorithmLoader::EvolutionaryAlgorithmLoader( DataLogger::DataLogger* log ):AlgorithmLoader::AlgorithmLoader( log )
{

}

/**
 * @see AlgorithmLoader
 * @return
 */
string EvolutionaryAlgorithmLoader::getNamespace() const
{
  return  "http://ailab.ifi.uzh.ch/testframework/2008/algorithm/evolution/";
}

/**
 * @see AlgorithmLoader
 * @param experiment
 * @return
 */
Algorithm* EvolutionaryAlgorithmLoader::load( const xmlpp::Element* experiment ) const
{
  EvolutionaryParam   param;

  initDefaultParam( &param );
  loadParam( getParamSection( experiment ), &param );

  return  new EvolutionaryAlgorithm
  (
      param.numberOfGenerations,
      param.populationSize,
      param.numberOfParents,
      param.changeProbability,
      param.elitismOn,
      param.parentSelection,
      param.distribution,
      param.parameterScaling,
      getLogger()
  );
}

/**
 * Loads the default values to param.
 * @param param
 */
void EvolutionaryAlgorithmLoader::initDefaultParam( EvolutionaryParam* param ) const
{
  param->numberOfParents      = EvolutionaryAlgorithm::DEFAULT_NUMBER_OF_PARENTS;
  param->numberOfGenerations  = EvolutionaryAlgorithm::DEFAULT_NUMBER_OF_GENERATIONS;
  param->populationSize       = EvolutionaryAlgorithm::DEFAULT_POPULATION_SIZE;
  param->elitismOn            = EvolutionaryAlgorithm::DEFAULT_ELITISM_ON;
  param->changeProbability    = 1.0 / EvolutionaryAlgorithm::DEFAULT_CHANGE_PROBABILITY_1_OVER;
  param->parentSelection      = EvolutionaryAlgorithm::DEFAULT_PARENT_SELECTION;
  param->distribution         = EvolutionaryAlgorithm::DEFAULT_DISTRIBUTION;
  param->parameterScaling     = 1.0 / EvolutionaryAlgorithm::DEFAULT_PARAMETER_SCALING_1_OVER;
}

/**
 * Reads the values from the configuration node to param.
 * @param config
 * @param param
 */
void EvolutionaryAlgorithmLoader::loadParam( const xmlpp::Element* config, EvolutionaryParam* param ) const
{
  if( config == NULL )
  {
    return;
  }

  param->numberOfParents      = strToInt( config->get_attribute_value( "parents", "" ) );
  param->numberOfGenerations  = strToInt( config->get_attribute_value( "generations", "" ) );
  param->populationSize       = strToInt( config->get_attribute_value( "population", "" ) );
  param->elitismOn            = strToOnOff( config->get_attribute_value( "elitism" ) );
  param->changeProbability    = strToFloat( config->get_attribute_value( "changeprobability", "" ) );
  param->parentSelection      = strToSelection( config->get_attribute_value( "selection", "" ) );
  param->distribution         = strToDistribution( config->get_attribute_value( "distribution", "" ) );
  param->parameterScaling     = strToFloat( config->get_attribute_value( "parameterscaling", "" ) );
}

/**
 * Returns the node which is containing the configuration.
 * @param experiment
 * @return
 */
const xmlpp::Element* EvolutionaryAlgorithmLoader::getParamSection( const xmlpp::Element* experiment ) const
{
  xmlpp::Node::NodeList nodes;
  xmlpp::Node::NodeList::iterator itr;

  nodes = experiment->get_children( "" );

  for( itr = nodes.begin(); itr != nodes.end(); itr++ )
  {
    if( ((*itr)->get_name() == "config") && ((*itr)->get_namespace_uri()
        == getNamespace()) )
    {
      return dynamic_cast<xmlpp::Element*> ( *itr );
    }
  }

  return NULL;
}
