/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "SimulatedAnnealingAlgorithm.h"

/**
 * Sets the whole bunch of member variables from the argument list. It also
 * initializes the random number generator.
 * @param initTemperature
 * @param terminateTemperature
 * @param cooling
 * @param bestFitness
 * @param boltzmannConstant
 * @param parameterScaling
 * @param log
 * @return
 */
SimulatedAnnealingAlgorithm::SimulatedAnnealingAlgorithm( double initTemperature, double terminateTemperature, double cooling, double bestFitness, double boltzmannConstant, double parameterScaling, DataLogger::DataLogger* log ): Algorithm::Algorithm( log )
{
  m_initTemperature       = initTemperature;
  m_terminateTemperature  = terminateTemperature;
  m_cooling               = cooling;
  m_bestFitness           = bestFitness;

  m_parameterScaling      = parameterScaling;
  m_boltzmannConstant     = boltzmannConstant;

  initRandomGenerator();
}

/**
 * Frees the random number generator.
 * @return
 */
SimulatedAnnealingAlgorithm::~SimulatedAnnealingAlgorithm()
{
  freeRandomGenerator();
}

/**
 * Create randomized parameters and reset the temperature.
 * @param changeable
 */
void SimulatedAnnealingAlgorithm::initialize( const MapParameterToRangeT* changeable )
{
  createParamsFromRange( changeable );

  randomizeParameters( 0 );
  randomizeParameters( 1 );
  randomizeParameters( 2 );

  m_oldFitness      = 0;
  m_temperature     = m_initTemperature;
  m_step            = 0;
}

/**
 * Best fitness nor terminating temperature reached
 * @return
 */
bool SimulatedAnnealingAlgorithm::hasNext() const
{
  return  (m_oldFitness < m_bestFitness) && (m_temperature > m_terminateTemperature);
}

/**
 * Business logic of the algorithm. Creates new values or uses the old ones.
 * @param fitness
 */
void SimulatedAnnealingAlgorithm::fitnessFeedback( double fitness )
{
   double deltaFitness;
   double p = 0.0;
   double r = 0.0;
   bool goBack = false;

   m_step++;
   deltaFitness = fitness - m_oldFitness;
   m_oldFitness = fitness;

   if( deltaFitness > 0.0 ) //value increased
   {
     m_temperature = m_cooling * m_temperature;
     newParameters( 1, fitness );
   }
   else
   {
     //      m_temperature = m_temperature; // remains the same :)
     p = exp( deltaFitness / (m_temperature * m_boltzmannConstant) );
     r = gsl_rng_uniform( m_randomGenerator );

     if( r < p )
     {
       newParameters( 1, fitness );
     }
     else
     {//we go back in the parameter landscape
       newParameters( 2, fitness );

       goBack = true;
     }
   }; //deltaV <=0


   if( goBack )
   {
     //params_Tmin2 = params_Tmin2; //because if went back, and next time we will need again to go back, we don't want to start from the point that was bad but rather from the original one
   }
   else
   {
     copyHistory( 1, 2 );
   }

   copyHistory( 0, 1 );
}

/**
 * Create and returns a map of the actual parameters.
 * @return
 */
MapParameterToValueT SimulatedAnnealingAlgorithm::next()
{
  MapParameterToValueT              result;
  list<SimanParameter>::iterator    itr;

  for( itr = m_param.begin(); itr != m_param.end(); itr++ )
  {
    result[itr->name]    = itr->value[0];
  }

  return  result;
}

/**
 * Randomize the parameter values.
 * @param historyIndex which values should be randomized
 */
void SimulatedAnnealingAlgorithm::randomizeParameters( int historyIndex )
{
  list<SimanParameter>::iterator     itr;

  for( itr = m_param.begin(); itr != m_param.end(); itr++ )
  {
    itr->value[historyIndex]   = itr->min + (gsl_rng_uniform( m_randomGenerator ) * (itr->max - itr->min));
  }
}

/**
 * Creates a new value for the parameter.
 * @param param
 * @param usedHistoryValue which old value is the base for the new value
 * @param fitness
 */
void SimulatedAnnealingAlgorithm::newParameter( SimanParameter* param, int usedHistoryValue, double fitness ) const
{
  double   n_t;
  double   a_i;

  do
  {
    do
    {
      n_t = gsl_ran_gaussian( m_randomGenerator, 1.0 );
    }
    while( (n_t > 1.0) || (n_t < -1.0) );

    a_i = n_t * (1 - (fitness / m_bestFitness));

    param->value[0]          = param->value[usedHistoryValue] + (a_i * m_temperature * (param->max - param->min) * m_parameterScaling);
  }
  while( (param->value[0] < param->min) || (param->value[0] > param->max) );
}

/**
 * Create new values for all parameters.
 * @param usedHistoryValue which old value is the base for the new value
 * @param fitness
 */
void SimulatedAnnealingAlgorithm::newParameters( int usedHistoryValue, double fitness )
{
  list<SimanParameter>::iterator        itr;

  for( itr = m_param.begin(); itr != m_param.end(); itr++ )
  {
    newParameter( &*itr, usedHistoryValue, fitness );   // wtf &* - because C++ is a very funny language
  };
}

/**
 * Allocate and initializes the random number generator.
 */
void SimulatedAnnealingAlgorithm::initRandomGenerator( void )
{
  gsl_rng_env_setup();

  m_randomGenerator   = gsl_rng_alloc( gsl_rng_default );
  gsl_rng_set( m_randomGenerator, rand() );
}

/**
 * Frees the random number generator.
 */
void SimulatedAnnealingAlgorithm::freeRandomGenerator()
{
  gsl_rng_free( m_randomGenerator );
}

/**
 * Copy the values of all parameter from one history index to an other.
 * @param from
 * @param to
 */
void SimulatedAnnealingAlgorithm::copyHistory( int from, int to )
{
  list<SimanParameter>::iterator    itr;

  for( itr = m_param.begin(); itr != m_param.end(); itr++ )
  {
    itr->value[to]    = itr->value[from];
  }
}

/**
 * Creates the internal parameter representation from a parameter range.
 * @param range
 */
void SimulatedAnnealingAlgorithm::createParamsFromRange( const MapParameterToRangeT* range )
{
  MapParameterToRangeT::const_iterator    itr;

  m_param.clear();

  for( itr = range->begin(); itr != range->end(); itr++ )
  {
    SimanParameter   param;

    param.max             = itr->second.maximum;
    param.min             = itr->second.minimum;
    param.name            = itr->first;
    param.value[0]        = (param.max + param.min) / 2;
    param.value[1]        = param.value[0];
    param.value[2]        = param.value[0];

    m_param.push_back( param );
  }
}

/**
 * Add a directory "step<n>".
 * @param dir
 */
void SimulatedAnnealingAlgorithm::extendDirectory( Directory& dir ) const
{
  dir.addSubdirectory( "step" + intToStr( m_step ) );
}

/**
 * @return
 */
string SimulatedAnnealingAlgorithm::getName() const
{
  return  "simulatedannealing";
}

