/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef SIMULATEDANNEALINGALGORITHM_H_
#define SIMULATEDANNEALINGALGORITHM_H_

#include  <list>
#include  <math.h>
#include  <gsl/gsl_rng.h>
#include  <gsl/gsl_randist.h>

#include  "../Algorithm.h"
#include  "../../../common/paramutils.h"
#include  "../../../common/convert.h"

/**
 * Contains data for one parameter
 */
typedef struct
{
  double      min;
  double      max;
  /** value history; index is age */
  double      value[3];
  string      name;
} SimanParameter;

/**
 * @class SimulatedAnnealingAlgorithm
 * @brief implements a simulated annealing
 */
class SimulatedAnnealingAlgorithm: public Algorithm
{
  private:
    double                  m_initTemperature;
    double                  m_terminateTemperature;
    double                  m_cooling;
    double                  m_bestFitness;

    /** Parameter to adjust the algorithm */
    double                  m_boltzmannConstant;
    /** Parameter to adjust the algorithm */
    double                  m_parameterScaling;

    double                  m_temperature;
    double                  m_oldFitness;
    list<SimanParameter>    m_param;
    int                     m_step;

    gsl_rng*                m_randomGenerator;

    void newParameter( SimanParameter* param, int historyIndex, double fitness ) const;
    void newParameters( int historyIndex, double val );
    void copyHistory( int from, int to );
    void createParamsFromRange( const MapParameterToRangeT* range );
    void randomizeParameters( int historyIndex );

    void initRandomGenerator( void );
    void freeRandomGenerator();

  public:
    SimulatedAnnealingAlgorithm( double initTemperature, double terminateTemperature, double cooling, double bestFitness, double boltzmannConstant, double parameterScaling, DataLogger::DataLogger* log );
    ~SimulatedAnnealingAlgorithm();

    void initialize( const MapParameterToRangeT* changeable );
    void fitnessFeedback( double fitness );
    bool hasNext() const;
    MapParameterToValueT next();
    void extendDirectory( Directory& dir ) const;

    string getName() const;
};

#endif /* SIMULATEDANNEALINGALGORITHM_H_ */
