/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file SystematicAlgorithm.cpp
 *
 * @brief Implements the SystematicAlgorithm.
 * @author Nicolas Rüegg, Urs Fässler
 */

#include "SystematicAlgorithm.h"

/**
 * Constructor.
 * @param steps	Default number of steps.
 * @param log		DataLogger to use for output.
 */
SystematicAlgorithm::SystematicAlgorithm( const map<string,int>* steps, DataLogger::DataLogger* log ): Algorithm::Algorithm( log )
{
  m_log             = log;

  map<string,int>::const_iterator  itr;

  for( itr = steps->begin(); itr != steps->end(); itr++ )
  {
    m_params[itr->first].stepCount   = itr->second;
  }
}


/**
 * Destructor.
 */
SystematicAlgorithm::~SystematicAlgorithm() {

}

/**
 * Initialize the algorithm. Reset all counters.
 * @param changeable	Map of changeable parameters.
 */
void SystematicAlgorithm::initialize( const MapParameterToRangeT* changeable )
{
  MapParameterToRangeT::const_iterator  itr;

  for( itr = changeable->begin(); itr != changeable->end(); itr++ )
  {
    m_params[itr->first].range    = itr->second;
    m_params[itr->first].step     = 0;
  }

  m_endReached  = false;
  m_step        = 0;
}

/**
 * Tests if there's another possible combination.
 *
 * @return true when another possible set of combinations is existing.
 */
bool SystematicAlgorithm::hasNext() const
{
  return  !( m_endReached );
}


/**
 * Calculate the value for a parameter.
 * @return value for the parameter.
 */
double SystematicAlgorithm::calcValue( const SystematicParameter& param ) const
{
  double    tmp;

  tmp   = param.range.maximum - param.range.minimum;
  tmp   = tmp * param.step;
  tmp   = tmp / (param.stepCount-1);
  tmp   = tmp + param.range.minimum;

  return  tmp;
}


/**
 * Sets the fitness value of the current set of parameters.
 * Has to be set by the experiment owning this algorithm based on the return
 * value returned by the experiment's run() method.
 *
 * @param fitness	Fitness value achieved by the current individual.
 */
void SystematicAlgorithm::fitnessFeedback( double fitness )
{
  map<string,SystematicParameter>::iterator   itr;

  m_fitness   = max( m_fitness, fitness );

  for( itr = m_params.begin(); itr != m_params.end(); itr++ )
  {
    itr->second.step++;

    if( itr->second.step >= itr->second.stepCount )
    {
      itr->second.step  = 0;
    }
    else
    {
      break;
    }
  }

  if( itr == m_params.end() )
  {
    m_endReached  = true;
  }
}


/**
 * Returns the next set of parameter value combinations.
 * @return	The next set of parameter values.
 */
MapParameterToValueT SystematicAlgorithm::next()
{
  MapParameterToValueT                        result;
  map<string,SystematicParameter>::iterator   itr;

  m_step++;

  for( itr = m_params.begin(); itr != m_params.end(); itr++ )
  {
    result[itr->first]   = calcValue( itr->second );
  }

  return  result;
}


/**
 * Append "/step<n>" to the directory.
 * @param dir	Current directory path.
 */
void SystematicAlgorithm::extendDirectory( Directory& dir ) const
{
  dir.addSubdirectory( "systematic" );
  dir.addSubdirectory( "step" + intToStr( m_step ) );
}

/**
 * @return
 */
string SystematicAlgorithm::getName() const
{
  return  "systematic";
}
