/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

#include  "TextbookSaAlgorithm.h"

/**
 * Sets the whole bunch of member variables from the argument list. It also
 * initializes the random number generator.
 * @param initTemperature
 * @param terminateTemperature
 * @param cooling
 * @param bestFitness
 * @param log
 * @return
 */
TextbookSaAlgorithm::TextbookSaAlgorithm( double initTemperature, double terminateTemperature, double cooling, double bestFitness, DataLogger::DataLogger* log ): Algorithm::Algorithm( log )
{
  m_initTemperature       = initTemperature;
  m_terminateTemperature  = terminateTemperature;
  m_cooling               = cooling;
  m_bestFitness           = bestFitness;

  initRandomGenerator();
}

/**
 * Frees the random number generator.
 * @return
 */
TextbookSaAlgorithm::~TextbookSaAlgorithm()
{
  freeRandomGenerator();
}

/**
 * Create randomized parameters and reset the temperature.
 * @param changeable
 */
void TextbookSaAlgorithm::initialize( const MapParameterToRangeT* changeable )
{
  createParamsFromRange( changeable );

//  m_error        = 1;
  m_step            = 0;
  m_temperature     = m_initTemperature;

  newParameters( 1 );
}

/**
 * Best fitness nor terminating temperature reached
 * @return
 */
bool TextbookSaAlgorithm::hasNext() const
{
  return  (m_errorOfOldNGood > 0) && (m_temperature > m_terminateTemperature);
}

/**
 * Business logic of the algorithm. Creates new values or uses the old ones.
 * @param fitness
 */
void TextbookSaAlgorithm::fitnessFeedback( double fitness )
{
   double error;
   double deltaError;
   double p;
   double r;


   m_step++;
   error        = 1 - (fitness / m_bestFitness);

   if( m_step == 1 )
   {
     m_errorOfOldNGood  = error;
     useNewValues();
     newParameters( 1 );
     return;
   }

   m_temperature = m_cooling * m_temperature;

   deltaError   = error - m_errorOfOldNGood;

   if( deltaError <= 0.0 )
   {
     p  = 1;
   }
   else
   {
     p  = exp( - m_initTemperature/m_temperature );
//     p  = exp( 1/m_initTemperature - 1/m_temperature );
//     p  = exp( -deltaError / m_temperature );
   }

   r    = gsl_rng_uniform( m_randomGenerator );

   if( r < p )
   {
     m_errorOfOldNGood  = error;
     useNewValues();
   }

   newParameters( m_temperature );
}

/**
 * Copys the values from index TSA_NEW to TSA_OLDNGOOD
 */
void TextbookSaAlgorithm::useNewValues()
{
  list<TextbookSaParameter>::iterator itr;

  for( itr = m_param.begin(); itr != m_param.end(); itr++ )
  {
    itr->value[TSA_OLDNGOOD]    = itr->value[TSA_NEW];
  }
}

/**
 * Create and returns a map of the actual parameters.
 * @return
 */
MapParameterToValueT TextbookSaAlgorithm::next()
{
  MapParameterToValueT                result;
  list<TextbookSaParameter>::iterator itr;

  for( itr = m_param.begin(); itr != m_param.end(); itr++ )
  {
    result[itr->name]    = itr->value[TSA_NEW];
  }

  return  result;
}

/**
 * Creates a new value for the parameter. The values are around the old good
 * values and scaled by scaling.
 * @param param
 * @param scaling
 */
void TextbookSaAlgorithm::newParameter( TextbookSaParameter* param, double scaling ) const
{
  double    x;
  double    rangeL;
  double    rangeH;
  double    range;

  range   = param->max - param->min;
  range   = range * scaling;

  rangeL  = param->value[TSA_OLDNGOOD] - range/2;
  rangeH  = param->value[TSA_OLDNGOOD] + range/2;

  rangeL  = max( rangeL, param->min );
  rangeH  = min( rangeH, param->max );

  range   = rangeH - rangeL;

  x       = gsl_rng_uniform( m_randomGenerator );
  x       = x * range;
  x       = x + rangeL;

  param->value[TSA_NEW]  = x;
}

/**
 * Create new values for all parameters.
 * @param scaling
 */
void TextbookSaAlgorithm::newParameters( double scaling )
{
  list<TextbookSaParameter>::iterator        itr;

  for( itr = m_param.begin(); itr != m_param.end(); itr++ )
  {
    newParameter( &*itr, scaling );   // wtf &* - because C++ is a very funny language
  };
}

/**
 * Allocate and initializes the random number generator.
 */
void TextbookSaAlgorithm::initRandomGenerator( void )
{
  gsl_rng_env_setup();

  m_randomGenerator   = gsl_rng_alloc( gsl_rng_default );
  gsl_rng_set( m_randomGenerator, rand() );
}

/**
 * Frees the random number generator.
 */
void TextbookSaAlgorithm::freeRandomGenerator()
{
  gsl_rng_free( m_randomGenerator );
}

/**
 * Creates the internal parameter representation from a parameter range.
 * @param range
 */
void TextbookSaAlgorithm::createParamsFromRange( const MapParameterToRangeT* range )
{
  MapParameterToRangeT::const_iterator    itr;

  m_param.clear();

  for( itr = range->begin(); itr != range->end(); itr++ )
  {
    TextbookSaParameter   param;

    param.max                 = itr->second.maximum;
    param.min                 = itr->second.minimum;
    param.name                = itr->first;
    param.value[TSA_OLDNGOOD] = (param.max + param.min) / 2;

    m_param.push_back( param );
  }
}

/**
 * Add a directory "step<n>".
 * @param dir
 */
void TextbookSaAlgorithm::extendDirectory( Directory& dir ) const
{
  dir.addSubdirectory( "step" + intToStr( m_step ) );
}

/**
 * @return
 */
string TextbookSaAlgorithm::getName() const
{
  return  "textbooksa";
}

