/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

#include  "TextbookSaAlgorithmLoader.h"



/**
 * @see AlgorithmLoader
 * @param log
 * @return
 */
TextbookSaAlgorithmLoader::TextbookSaAlgorithmLoader( DataLogger::DataLogger* log ):AlgorithmLoader::AlgorithmLoader( log )
{
}

/**
 * @see AlgorithmLoader
 * @return
 */
string TextbookSaAlgorithmLoader::getNamespace() const
{
  return  "http://vidyaa.origo.ethz.ch/2009/algorithm/textbooksa/";
}

/**
 * @see AlgorithmLoader
 * @param experiment
 * @return
 */
Algorithm* TextbookSaAlgorithmLoader::load( const xmlpp::Element* experiment ) const
{
  TextbookSaLoaderParam   param;

  loadParam( getConfigSection( experiment ), &param );
  verifyParam( &param );

  return  new TextbookSaAlgorithm
  (
      param.initTemperature,
      param.terminateTemperature,
      param.cooling,
      param.bestFitness,
      getLogger()
  );
}

/**
 * Loads and returns the parameter for the algorithm.
 * @param config configuration node
 * @param param parsed parameter
 */
void TextbookSaAlgorithmLoader::loadParam( const xmlpp::Element* config, TextbookSaLoaderParam* param ) const
{
  if( config == NULL )
  {
    throw new LoaderException( "Simulated Annealing needs parameter" );
  }

  param->bestFitness            = strToFloat( config->get_attribute_value( "bestfitness", "" ) );
  param->cooling                = strToFloat( config->get_attribute_value( "cooling", "" ) );
  param->initTemperature        = strToFloat( config->get_attribute_value( "inittemperature", "" ) );
  param->terminateTemperature   = strToFloat( config->get_attribute_value( "terminatetemperature" ) );
}

/**
 * Returns the configuration node.
 * @param experiment
 * @return
 */
const xmlpp::Element* TextbookSaAlgorithmLoader::getConfigSection( const xmlpp::Element* experiment ) const
{
  xmlpp::Node::NodeList nodes;
  xmlpp::Node::NodeList::iterator itr;

  nodes = experiment->get_children( "" );

  for( itr = nodes.begin(); itr != nodes.end(); itr++ )
  {
    if( ((*itr)->get_name() == "config") && ((*itr)->get_namespace_uri() == getNamespace()) )
    {
      return dynamic_cast<xmlpp::Element*> ( *itr );
    }
  }

  return NULL;
}

/**
 * Validates the parameter. Throws a LoaderException if a error occurs.
 * @param param
 */
void TextbookSaAlgorithmLoader::verifyParam( const TextbookSaLoaderParam* param ) const
{
  if( param->bestFitness <= 0.0 )
  {
    throw new LoaderException( "TextbookSa: bestfitness <= 0 found" );
  }

  if( (param->cooling <= 0.0) || (param->cooling >= 1.0) )
  {
    throw new LoaderException( "TextbookSa: cooling not in (0,1)" );
  }

  if( param->terminateTemperature <= 0.0 )
  {
    throw new LoaderException( "TextbookSa: terminatetemperature <= 0 found" );
  }

  if( param->initTemperature < param->terminateTemperature )
  {
    throw new LoaderException( "TextbookSa: inittemperature < terminatetemperature found" );
  }
}

