/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Experiment.h"

/**
 * set the datalogger
 * @param log
 * @return
 */
Experiment::Experiment( FitnessCalculation fitness, DataLogger::DataLogger *log )
{
  m_log             = log;
  m_fitnessType     = fitness;

  m_fitness.count   = 0;
  m_fitness.max     = numeric_limits<double>::min();
  m_fitness.min     = numeric_limits<double>::max();
  m_fitness.product = 1;
  m_fitness.sum     = 0;
}

/**
 * Adds a new computed fitness value. The final fitness value is calculated at
 * at the end of a experiment.
 * @param fitness
 */
void Experiment::addFitness( double fitness, const MapParameterToValueT* values )
{
  switch( m_fitnessType )
  {
    case FC_PRODUCT:
    case FC_INVERSEPRODUCT:
    case FC_SUM:
    case FC_AVERAGE:        break;
    case FC_MINIMUM:        if( fitness < m_fitness.min ){ m_fitness.map = *values; } break;
    case FC_MAXIMUM:        if( fitness > m_fitness.max ){ m_fitness.map = *values; } break;
    default:                throw new LoaderException( "Unknown value of FitnessCalculation in Experiment::addFitness" );
  }

  m_fitness.count++;
  m_fitness.max     = max( m_fitness.max, fitness );
  m_fitness.min     = min( m_fitness.min, fitness );
  m_fitness.sum     = m_fitness.sum + fitness;
  m_fitness.product = m_fitness.product * fitness;
}

void Experiment::writeResult( string filename, double fitness ) const
{
  ofstream    file;
  string      sep = " ";

  file.open( filename.c_str(), ios::out );

  file << "fitness" << sep << fitness << endl;
  file << "calculation" << sep << serializeFitnessCalculation( m_fitnessType ) << endl;
  file << "count" << sep << m_fitness.count << endl;
  file << "max" << sep << m_fitness.max << endl;
  file << "min" << sep << m_fitness.min << endl;
  file << "sum" << sep << m_fitness.sum << endl;
  file << "product" << sep << m_fitness.product << endl;

  file.close();
}

void Experiment::writeMap( string filename ) const
{
  if( (m_fitnessType != FC_MAXIMUM) && (m_fitnessType != FC_MINIMUM) )
  {
    return;
  }
  ofstream    file;

  file.open( filename.c_str(), ios::out );
  serializeMap( m_fitness.map, file );
  file.close();
}

/**
 * Finalise the experiment. Returns the fitness value and write some files.
 * @return
 */
double Experiment::exitExperiment()
{
  string      filename;
  double      fitness;

  m_directory.addSubdirectory( getName() );

  m_directory.setIncludeFilename( true );
  m_directory.createDirectory();
  filename  = m_directory.getDirectory();

  switch( m_fitnessType )
  {
    case FC_AVERAGE:        fitness = m_fitness.sum / m_fitness.count; break;
    case FC_MINIMUM:        fitness = m_fitness.min; break;
    case FC_MAXIMUM:        fitness = m_fitness.max; break;
    case FC_PRODUCT:        fitness = m_fitness.product; break;
    case FC_INVERSEPRODUCT: fitness = 1 / m_fitness.product; break;
    case FC_SUM:            fitness = m_fitness.sum; break;
    default:                throw new LoaderException( "Unknown value of FitnessCalculation in Experiment::exitExperiment" );
  }

  getLogger()->write( RESULT, (string)"file" + DataLogger::SEPERATOR + filename + ".parameter" + DataLogger::SEPERATOR + "fitness" + DataLogger::SEPERATOR + floatToStr( fitness ) );

  writeResult( filename + ".fitness", fitness );
  writeMap( filename + ".parameter" );

  return  fitness;
}

/**
 * returns the datalogger
 * @return
 */
DataLogger::DataLogger* Experiment::getLogger()
{
  return  m_log;
}


/**
 * Parse value for values of FitnessCalculation and throws a LoaderException
 * if a error occurs.
 * @param value
 * @return
 */
FitnessCalculation Experiment::parseFitnessCalculation( string value )
{
  map<string,FitnessCalculation>    EF_NAME;
  map<string,FitnessCalculation>::iterator    idx;

  EF_NAME["average"]        = FC_AVERAGE;
  EF_NAME["minimum"]        = FC_MINIMUM;
  EF_NAME["maximum"]        = FC_MAXIMUM;
  EF_NAME["product"]        = FC_PRODUCT;
  EF_NAME["inverseproduct"] = FC_INVERSEPRODUCT;
  EF_NAME["sum"]            = FC_SUM;

  idx = EF_NAME.find( value );
  if( idx == EF_NAME.end() )
  {
    throw new LoaderException( "fitness calculation type not found (" + value + ")" );
  }

  return  idx->second;
}

/**
 * Returns a string out of a FitnessCalculation value.
 * @param calculation
 * @return
 */
string Experiment::serializeFitnessCalculation( FitnessCalculation calculation )
{
  switch( calculation )
  {
    case FC_AVERAGE:        return  "average";
    case FC_MINIMUM:        return  "minimum";
    case FC_MAXIMUM:        return  "maximum";
    case FC_PRODUCT:        return  "product";
    case FC_INVERSEPRODUCT: return  "inverseproduct";
    case FC_SUM:            return  "sum";
    default:                throw new LoaderException( "Unknown value of FitnessCalculation in Experiment::serializeFitnessCalculation" );
  }
}

