/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * experimentloader.cpp
 *
 *  Created on: 13.04.2009
 *      Author: urs
 */


#include  "ExperimentLoader.h"

/**
 * set m_loader
 * @param loader
 */
void AbstractExperimentLoader::setLoader( ExperimentLoader* loader )
{
  m_loader    = loader;
}

/**
 * returns m_loader
 * @return
 */
ExperimentLoader* AbstractExperimentLoader::getLoader() const
{
  return  m_loader;
}

/**
 * Returns the FitnessCalculation value or throws a LoaderException if it is
 * not found.
 * @param experiment
 * @return
 */
FitnessCalculation AbstractExperimentLoader::readFitness( const xmlpp::Element* experiment ) const
{
  return  readFitness( experiment, FC_AVERAGE, false );
}

/**
 * Returns the FitnessCalculation value or the value of default_ if it is
 * not found.
 * @param experiment
 * @param default_
 * @return
 */
FitnessCalculation AbstractExperimentLoader::readFitness( const xmlpp::Element* experiment, FitnessCalculation default_ ) const
{
  return  readFitness( experiment, default_, true );
}

/**
 * Returns the FitnessCalculation value. If it is not found, a LoaderException
 * is thrown when useDefault is false or the value of default_ is returned when
 * useDefault is true.
 * @param experiment
 * @param default_
 * @param useDefault
 * @return
 */
FitnessCalculation AbstractExperimentLoader::readFitness( const xmlpp::Element* experiment, FitnessCalculation default_, bool useDefault ) const
{
  xmlpp::Attribute*   attr;

  attr    = experiment->get_attribute( "fitness", "" );
  if( attr != NULL )
  {
    return  Experiment::parseFitnessCalculation( attr->get_value() );
  }
  else
  {
    if( useDefault )
    {
      return  default_;
    }
    else
    {
      throw new LoaderException( "fitness attribute not found! (" + experiment->get_name() + ")" );
    }
  }
}

//-----------------------------------------------------------------------------


/**
 * Frees all loaders.
 */
ExperimentLoader::~ExperimentLoader()
{
  map<string,const AbstractExperimentLoader*>::const_iterator   idx;

  for( idx = m_loaders.begin(); idx != m_loaders.end(); idx++ )
  {
    delete idx->second;
  }
}

/**
 * Register a new ExperimentLoader under the given name. The ExperimentLoader
 * takes the ownership of the loaders.
 * @param name
 * @param loader
 */
void ExperimentLoader::addExperimentLoader( string name, AbstractExperimentLoader* loader )
{
  loader->setLoader( this );
  m_loaders[name]   = loader;
}

/**
 * Returns the specified ExperimentLoader or throws a LoaderException if the
 * loader is unknown.
 * @param name
 * @return the experiment loader
 */
const AbstractExperimentLoader* ExperimentLoader::getExperimentLoader( string name ) const
{
  map<string,const AbstractExperimentLoader*>::const_iterator   idx;

  idx   = m_loaders.find( name );

  if( idx == m_loaders.end() )
  {
    throw new LoaderException( "ExperimentLoader \"" + name + "\" not found" );
  }

  return   idx->second;
}

/**
 * Returns the child node which contains a experiment.
 * @param root
 * @return the experiment node or NULL
 */
const xmlpp::Element* ExperimentLoader::findExperiment( const xmlpp::Element* root ) const
{
  map<string,const AbstractExperimentLoader*>::const_iterator   itr;

  for( itr = m_loaders.begin(); itr != m_loaders.end(); itr++ )
  {
    string debug   = itr->first;
    const xmlpp::Node::NodeList           nodes       = root->get_children( itr->first );

    if( nodes.size() == 1 )
    {
      return  dynamic_cast<xmlpp::Element*>( *nodes.begin() );
    }
  }

  return  NULL;
}

/**
 * Loads and returns a experiment of the specified configuration. Throws a
 * LoaderException if an error occurs.
 * @param experiment
 * @return a new instance of the experiment (with all sub experiments)
 */
Experiment* ExperimentLoader::createExperiment( const xmlpp::Element* experiment ) const
{
  if( experiment != NULL )
  {
    const AbstractExperimentLoader*   loader;
    string                      name;
    const char*                 debug;

    name      = experiment->get_name();
    debug     = name.c_str();
    loader    = getExperimentLoader( name );

    return  loader->createExperiment( experiment );
  }

  throw new LoaderException( "no experiment/simulation/array defined (or more than one)" );
}

/**
 * Returns the settings for all experiments (from root node).
 * @param experiment
 * @param settings
 */
void ExperimentLoader::readSettings( const xmlpp::Element* documentNode, ExperimentSettings* settings ) const
{
  xmlpp::Attribute*   attr;

  attr  = documentNode->get_attribute( "seed", "" );
  if( attr != NULL )
  {
    settings->seed  = strToInt( attr->get_value() );
  }
}

/**
 * returns m_logger
 * @return
 */
DataLogger::DataLogger* ExperimentLoader::getLogger() const
{
  return  m_logger;
}

/**
 * set m_logger
 * @param logger
 */
void ExperimentLoader::setLogger( DataLogger::DataLogger* logger )
{
  m_logger  = logger;
}

