/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * experimentLoader.h
 *
 *  Created on: 13.04.2009
 *      Author: urs
 */

#ifndef EXPERIMENTLOADER_H_
#define EXPERIMENTLOADER_H_

#include  <libxml++/libxml++.h>
#include  <libxml++/nodes/node.h>
#include  <string>


#include  "../common/convert.h"

#include  "LoaderException.h"
#include  "Experiment.h"
#include  "../DataLogger/DataLogger.h"

typedef struct
{
  long    seed;
} ExperimentSettings;

class ExperimentLoader;

/**
 * @class AbstractExperimentLoader
 * @brief Base class for all experiment loaders
 *
 * A implementation of AbstractExperimentLoader got a experiment configuration
 * as a xml node and creates a experiment object.
 */
class AbstractExperimentLoader
{
  private:
    ExperimentLoader*   m_loader;
    FitnessCalculation readFitness( const xmlpp::Element* experiment, FitnessCalculation default_, bool useDefault ) const;

  protected:
    FitnessCalculation readFitness( const xmlpp::Element* experiment ) const;
    FitnessCalculation readFitness( const xmlpp::Element* experiment, FitnessCalculation default_ ) const;

  public:
    virtual Experiment* createExperiment( const xmlpp::Element* experiment ) const = 0;

    void setLoader( ExperimentLoader* loader );
    ExperimentLoader* getLoader() const;
};

/**
 * @class ExperimentLoader
 * @brief Creates a experiment from a configuration
 *
 * If a experiment should be created, the loader for this experiment is
 * searched. For this, experiment loaders can be added.
 *
 * @see AbstractExperimentLoader
 */
class ExperimentLoader
{
  private:
    DataLogger::DataLogger*                        m_logger;
    map<string,const AbstractExperimentLoader*>    m_loaders;
    bool isInList( string name );

  public:
    ~ExperimentLoader();

    const xmlpp::Element* findExperiment( const xmlpp::Element* root ) const;
    Experiment* createExperiment( const xmlpp::Element* experiment ) const;
    void readSettings( const xmlpp::Element* documentNode, ExperimentSettings* settings ) const;

    void addExperimentLoader( string name, AbstractExperimentLoader* loader );
    const AbstractExperimentLoader* getExperimentLoader( string name ) const;
    DataLogger::DataLogger* getLogger() const;
    void setLogger( DataLogger::DataLogger* logger );
};


#endif /* EXPERIMENTLOADER_H_ */
