/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file SimulationExperiment.cpp
 *
 * @brief Implementation of the SimulationExperiment.
 */


#include  "SimulationExperiment.h"

/**
 * The constructer.
 *
 * Creates a SimulationExperiment.
 *
 * @param program	Path to the simulation program.
 * @param argument	Additional arguments to the simulation program.
 * @param log		DataLogger for data output.
 */
SimulationExperiment::SimulationExperiment( string program, string argument, FitnessCalculation fitness, DataLogger::DataLogger* log ) : Experiment::Experiment( fitness, log )
{
  m_program    = program;
  m_argument    = argument;
}

/**
 * Destructor.
 */
SimulationExperiment::~SimulationExperiment() {

}

/**
 * @return
 */
string SimulationExperiment::getName() const
{
  return  "simulation";
}

/**
 * Runs the SimulationExperiment.
 *
 * Creates a file [INPUTFILE] with the parameter names and their values for the simulation
 * software.
 * Calls the simulation program with the arguments "-i [INPUTFILE] -o [OUTPUTFILE]"
 * and the additional arguments specified.
 * Reads the results of the simulation from the file [OUTPUTFILE] and returns the
 * fitness value.
 *
 * @param parameterValues	Set of parameters with their values to simulate.
 * @param path				The directory where the results are written to.
 * @return	The fitness value of this simulation.
 */
double SimulationExperiment::run( const MapParameterToValueT* parameterValues, Directory path )
{
  path.setIncludeFilename( true );
  path.createDirectory();

  string    filename  = path.getDirectory();
  string    outFile   = filename + ".parameter";
  string    inFile    = filename + ".result";
  string    cmd;
  fstream   file;

  cmd   = m_program + " -i " + outFile + " -o " + inFile + " " + m_argument;

  file.open( outFile.c_str(), ios::trunc | ios::out );
  serializeMap( *parameterValues, file );
  file.close();

  getLogger()->write( INFO, "simulating file: " + outFile );

  if( system( cmd.c_str() ) < 0 )
  {
    //TODO: hier ist rückgabewert von system. man müsste noch rückgabewert vom simulator testen
    throw new runtime_error( "SimulationExperiment: simulator returner error" );
  }

  MapParameterToValueT  result;

  file.open( inFile.c_str(), ios::in );
  deserializeMap( result, file );
  file.close();

  getLogger()->write( parameterValues, &result );

  return  getFitness( &result );
}


/**
 * Looks for a fitness value (identified by "fitness") in a map an returns its value.
 * Throws a runtime_error exception if not found.
 *
 * @param result	Map where it should look for a fitness value.
 * @return The fitness value.
 */
double SimulationExperiment::getFitness( const map<string,double>* result ) const
{
  map<string,double>::const_iterator    idx;

  idx   = result->find( "fitness" );

  if( idx == result->end() )
  {
    throw new runtime_error( "Fitness value in result not found" );
  }

  return  idx->second;
}
