/*
    Copyright 2009 Nicolas Rüegg, Urs Fässler


    This file is part of Vidyaa.

    Vidyaa is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Vidyaa is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Vidyaa.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file main.cpp
 *
 * @brief Main
 */

#include  <argp.h>
#include <iostream>
#include "Experimenter.h"

using namespace std;

const char *argp_program_version =      "vidyaa Version 0.1";
const char *argp_program_bug_address =  "<nobug@thisapp.unv>";

/* Program documentation. */
static char doc[] =     "vidyaa: Tool for the optimization of parameter values.";

/* A description of the arguments we accept. */
static char args_doc[] = "[FILE]";

#define     SWITCH_EXPERIMENTS    'e'
#define     SWITCH_LOGFILE        'l'

#define     SWITCH_LOG_RESULT       'R'
#define     SWITCH_LOG_INFO         'I'
#define     SWITCH_LOG_WARNING      'W'
#define     SWITCH_LOG_ERROR        'E'
#define     SWITCH_LOG_NOTHING      'N'

/* The options we understand. */
static struct argp_option options[] =
{
  { "experiments",      SWITCH_EXPERIMENTS,    "DIR",  0,  "Root path to which experiment results are written." },
  { "logfile",          SWITCH_LOGFILE,        "FILE", 0,  "Write logfile" },
  {0,0,0,0, "The following arguments control the log level. By default, all arguments are set. If one (or more) option is specified, the default value is overwritten:" },
  { "nothing",          SWITCH_LOG_NOTHING,    0,      0,  0 },
  { "infos",            SWITCH_LOG_INFO,       0,      0,  0 },
  { "results",          SWITCH_LOG_RESULT,     0,      0,  0 },
  { "warnings",         SWITCH_LOG_WARNING,    0,      0,  0 },
  { "errors",           SWITCH_LOG_ERROR,      0,      0,  0 },
  { 0 }
};

/* Used by main to communicate with parse_opt. */
struct Arguments
{
  string*  dirExperiments;
  string*  fileLog;
  string*  fileConfig;
  int      logbit;
};

/* Parse a single option. */
static error_t parse_opt (int key, char *arg, struct argp_state *state)
{
 /* Get the input argument from argp_parse, which we
    know is a pointer to our arguments structure. */
  struct Arguments *arguments = (struct Arguments*)state->input;

  switch (key)
  {
    case SWITCH_LOG_NOTHING:
    {
      if( arguments->logbit < 0 )
      {
        arguments->logbit = 0;
      }
      arguments->logbit = arguments->logbit | NONE;
      break;
    }

    case SWITCH_LOG_RESULT:
    {
      if( arguments->logbit < 0 )
      {
        arguments->logbit = 0;
      }
      arguments->logbit = arguments->logbit | RESULT;
      break;
    }

    case SWITCH_LOG_INFO:
    {
      if( arguments->logbit < 0 )
      {
        arguments->logbit = 0;
      }
      arguments->logbit = arguments->logbit | INFO;
      break;
    }

    case SWITCH_LOG_WARNING:
    {
      if( arguments->logbit < 0 )
      {
        arguments->logbit = 0;
      }
      arguments->logbit = arguments->logbit | WARN;
      break;
    }

    case SWITCH_LOG_ERROR:
    {
      if( arguments->logbit < 0 )
      {
        arguments->logbit = 0;
      }
      arguments->logbit = arguments->logbit | ERROR;
      break;
    }

    case SWITCH_EXPERIMENTS:
    {
      arguments->dirExperiments = new string( arg );
      break;
    }

    case SWITCH_LOGFILE:
    {
      arguments->fileLog = new string( arg );
      break;
    }

    case ARGP_KEY_ARG:
    {
      if( state->arg_num != 0 )
      {
        argp_usage (state);
      }
      arguments->fileConfig = new string( arg );

      break;
    }

    case ARGP_KEY_END:
    {
      if( arguments->fileConfig == NULL )
      {
        argp_usage (state);
      }

/*      if (state->arg_num < 2)
      {
        // Not enough arguments.
        argp_usage (state);
      }*/
      break;
    }

    default:
    {
      return ARGP_ERR_UNKNOWN;
    }
  }

  return 0;
}

/* Our argp parser. */
static struct argp argp = { options, parse_opt, args_doc, doc };

int main(int argc, char* argv[])
{
  struct Arguments arguments;

  /* Default values. */
  arguments.dirExperiments = NULL;
  arguments.fileConfig     = NULL;
  arguments.fileLog        = NULL;
  arguments.logbit         = -1;

  /* Parse our arguments; every option seen by parse_opt will
     be reflected in arguments. */
  argp_parse (&argp, argc, argv, ARGP_IN_ORDER, 0, &arguments);

  if( arguments.dirExperiments == NULL )
  {
    arguments.dirExperiments  = new string( "./" );
  }

  try
  {

		// print welcome message
		cout << "\n------------------------------------------------------" << endl;
//		cout << "Optimization Framework, AI Lab University of Zurich" << endl;
		cout << doc << endl;
		cout << "Optimization made easy." << endl;
		cout << "------------------------------------------------------\n" << endl;

		cout << "Starting experiment " << *arguments.fileConfig << "... " << endl;

		// Instantiate the Experimenter. That's the class who finally conducts the experiments
		cout << "Instantiating Experimenter... ";
		Experimenter *experimenter = new Experimenter( *arguments.fileConfig, *arguments.fileLog, arguments.logbit );
		cout << "[done]" << endl;

		// Run the experiment.

	  cout << "Running Experiment..." << endl;
    experimenter->run( Directory( *arguments.dirExperiments, true ) );
	  cout << "run finished" << endl;

		cout << "Experiment finished... Results were logged to " << *arguments.fileLog << "." << endl;
		cout << "Cleaning up... ";
		delete experimenter;
		cout << "[done]" << endl;
  }
  catch( LoaderException* e )
  {
    cerr << "Error: " << e->getMsg() << endl;
  }
  catch( runtime_error* e )
  {
    cerr << "Error: " << e->what() << endl;
  }

	return 0;
}





